/* -*-c++-*- 
 * This source code is proprietary of ADIT
 * Copyright (C) 2013 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Vadiraj Kaamsha <vadiraj.kaamsha@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

#include <osgBatchedText/TextSimpleBase>
#include <osgBatchedText/BatcherBase>

using namespace osgBatchedText;

TextSimpleBase::TextSimpleBase():
    _text(NULL),
    _charOffset(),
    _radius(0.0f),
    _verticalOffset3D(0.0f),
    _relCharPosScaleFactor(0.0f),
    _has_position(false),
    _glyphRepresentationCalculated(false),
    _displayOrder(0),
    _hasCharCodes(true),
    _numDrawableChars(0)
{
}

TextSimpleBase::TextSimpleBase(const TextSimpleBase& src) : Referenced(src),
    _text(NULL),
    _charOffset(src._charOffset),
    _position(src._position),
    _radius(src._radius),
    _verticalOffset3D(src._verticalOffset3D),
    _relCharPosScaleFactor(src._relCharPosScaleFactor),
    _relCharPosition(src._relCharPosition),
    _vertices(src._vertices),
    _charCodesOrGlyphIndices(src._charCodesOrGlyphIndices),
    _textConfigBase(src._textConfigBase),
    _textBB(src._textBB),
    _has_position(src._has_position),
    _glyphRepresentationCalculated(src._glyphRepresentationCalculated),
    _displayOrder(src._displayOrder),
    _hasCharCodes(src._hasCharCodes),
    _numDrawableChars(src._numDrawableChars)
{
   setText(src._text);
}


TextSimpleBase::~TextSimpleBase()
{
   delete[] _text;
   _text = NULL;
}

void TextSimpleBase::computeGlyphRepresentation(GlyphInfoContainerBase* glyphInfoContainer, bool useVerticalLayout)
{
   static const half_float::half zeroHalf(0.0f);

   std::vector<osg::Vec2> positionOffset;
   std::vector<LineInfo> lineInfo;
   std::vector<osg::Vec2> vertices;
   if (computeGlyphRepresentationCore(glyphInfoContainer, positionOffset, lineInfo, vertices, useVerticalLayout))
   {
      _vertices.clear();
      _vertices.reserve(vertices.size());

      for (unsigned int countVert = 0; countVert < vertices.size(); ++countVert)
      {
         vertices[countVert] += positionOffset[countVert/4];
         _vertices.push_back(osg::Vec3h(vertices[countVert].x(), vertices[countVert].y(), zeroHalf));
      }

      _glyphRepresentationCalculated = true;
   }
}

unsigned int TextSimpleBase::splitIntoStringPerLine(const char* inputText, std::vector<std::string>& outputStrings) const
{
   static const char CR = '\n';
   std::string part;
   std::size_t start = 0;
   unsigned int lineInfoSize = 1;

   std::string inputString(inputText);
   std::size_t found = inputString.find(CR);
   while (found != std::string::npos)
   {
      part = inputString.substr(start, found - start);
      outputStrings.push_back(part);

      ++lineInfoSize;

      start = found + 1;
      found = inputString.find(CR, start);
   }

   //Last part
   part = inputString.substr(start);
   outputStrings.push_back(part);

   return lineInfoSize;
}

bool TextSimpleBase::computeGlyphVertices(const ComputeVerticesInput& input,
                                          GlyphInfoContainerBase* glyphInfoContainer, std::vector<osg::Vec2>& positionOffset,
                                          std::vector<LineInfo>& lineInfo, std::vector<osg::Vec2>& vertices)
{
   if (!(input._utf8Strings) && !(input._glyphInfoVector))
   {
      return false;
   }

   //configuration parameters
   float fontSize = static_cast<float>(_textConfigBase->getFontSize());
   float lineSpacing = _textConfigBase->getLineSpacing();
   float characterHeight = _textConfigBase->getCharSize();
   float scale = characterHeight / fontSize;
   BatchDrawableBase::tShaderType shaderType = _textConfigBase->getShaderType();

   //clear
   positionOffset.clear();
   lineInfo.clear();
   vertices.clear();
   _charCodesOrGlyphIndices.clear();
   _textBB.init();

   //reserve space
   _numDrawableChars = 0;
   _charCodesOrGlyphIndices.reserve(input._charCodeOrGlyphCount);
   positionOffset.reserve(input._charCodeOrGlyphCount);
   vertices.reserve(4 * input._charCodeOrGlyphCount);
   lineInfo.reserve((input._layout == osgText::TextBase::VERTICAL) ? input._charCodeOrGlyphCount : input._lineInfoSize);

   osg::Vec2 position;
   unsigned int lineCount = 0;
   bool useUnicodeStrings = (input._utf8Strings != NULL);
   const GlyphInfoContainerBase::LineBreakCharCodesOrGlyphIndices& lineBreakCharCodesOrGlyphIndices = glyphInfoContainer->getLineBreakCharCodesOrGlyphIndices(useUnicodeStrings);

   for (unsigned int lineIndex = 0; lineIndex < input._lineInfoSize; ++lineIndex)
   {
      if (lineIndex > 0)
      {
         ++lineCount;
         position.x() = 0.0f;
         position.y() -= characterHeight*(1.0 + lineSpacing);
      }

      osgText::glyphEntryUpper upper;
      osg::Vec2 delta;
      osg::Vec2 tl;
      osg::Vec2 br;
      unsigned int prevglyphindex = 0xFFFFFFFF;

      osg::BoundingBox lineBB;
      osgText::String unicodeString;
      const std::vector<GlyphInfo> *glyphInfo = NULL;
      unsigned int count = 0;

      if (useUnicodeStrings)
      {
         unicodeString.set((*(input._utf8Strings))[lineIndex], osgText::String::ENCODING_UTF8);
         if (input._layout == osgText::TextBase::RIGHT_TO_LEFT)
         {
            std::reverse(unicodeString.begin(), unicodeString.end());
         }
         count = unicodeString.size();
      }
      else
      {
         glyphInfo = &((*(input._glyphInfoVector))[lineIndex]);
         count = (*glyphInfo).size();
      }

      for (unsigned int i = 0; i < count; ++i)
      {
         unsigned int charCodeOrGlyphIndex = (glyphInfo ? (*glyphInfo)[i]._glyphIndex : unicodeString[i]);

         if (glyphInfoContainer->getGlyph(useUnicodeStrings, charCodeOrGlyphIndex, upper, prevglyphindex, delta, false, useUnicodeStrings))
         {
            osg::Vec2 advance;
            if (glyphInfo)
            {
               upper._position.x() += (float)((*glyphInfo)[i]._xOffset);
               upper._position.y() += (float)((*glyphInfo)[i]._yOffset);

               advance.x() = (float)((*glyphInfo)[i]._xAdvance);
               advance.y() = (float)((*glyphInfo)[i]._yAdvance);
            }
            else
            {
               advance = upper._advance;
            }

            float centerOffset = upper._position.x() + (upper._dimension.x() * 0.5f);
            upper._position.x() -= centerOffset;

            tl = upper._position * scale;
            br.x() = tl.x() + (upper._dimension.x() * scale);
            br.y() = tl.y() + (upper._dimension.y() * scale);

            bool alignCenterOfChar = ((shaderType == BatchDrawableBase::CURVED_SCREEN) || (shaderType == BatchDrawableBase::CURVED_GROUND));

            if (!alignCenterOfChar)
            {
               position.x() += centerOffset * scale;
               position += delta * (scale * fontSize);
            }

            _charCodesOrGlyphIndices.push_back(charCodeOrGlyphIndex);
            if (charCodeOrGlyphIndex != lineBreakCharCodesOrGlyphIndices._space)
            {
               vertices.push_back(osg::Vec2(tl.x(), br.y())); /* bottom left  */
               vertices.push_back(br);                        /* bottom right */
               vertices.push_back(osg::Vec2(br.x(), tl.y())); /* top right    */
               vertices.push_back(tl);                        /* top left     */

               positionOffset.push_back(position);
               ++_numDrawableChars;
            }

            lineBB.expandBy(osg::Vec3(tl + position, 0.0f)); //top left corner
            lineBB.expandBy(osg::Vec3(br + position, 0.0f)); //bottom right corner

            if (input._layout == osgText::TextBase::VERTICAL)
            {
               position.y() += advance.y() * scale;

               _textBB.expandBy(lineBB);
               lineInfo.push_back(LineInfo(lineBB.xMin(), lineBB.xMax(), _numDrawableChars));
               lineBB.init();
               ++lineCount;
            }
            else
            {
               position.x() += alignCenterOfChar ? characterHeight : (advance.x() - centerOffset) * scale;
            }
         }
      }

      /* position correction */
      _textBB.expandBy(lineBB);
      if (input._layout != osgText::TextBase::VERTICAL)
      {
         lineInfo.push_back(LineInfo(lineBB.xMin(), lineBB.xMax(), _numDrawableChars));
      }
   }

   unsigned int index = 0;
   osg::Vec2 offset(0.0f, calcBottomOffset(lineCount));
   //maximum length of the line for text bounding box adjustment
   float maxLineLength = 0.0f;
   float xOffsetOfBB = 0.0f;
   for (std::vector<LineInfo>::iterator itrPos = lineInfo.begin(); itrPos != lineInfo.end(); ++itrPos)
   {
      offset.x() = calcLeftOffset(*itrPos);
      if (maxLineLength < itrPos->_xMax - itrPos->_xMin)
      {
         maxLineLength = itrPos->_xMax - itrPos->_xMin;
         xOffsetOfBB = offset.x();
      }
      //adjust position offsets for the line
      while (index < itrPos->_numChars)
      {
         positionOffset[index++] += offset;
      }
   }
   //adjust text bounding box
   _textBB.xMin() += xOffsetOfBB;
   _textBB.xMax() += xOffsetOfBB;
   _textBB.yMin() += offset.y();
   _textBB.yMax() += offset.y();

   return true;
}

bool TextSimpleBase::computeGlyphRepresentationCore(GlyphInfoContainerBase* glyphInfoContainer, std::vector<osg::Vec2>& positionOffset, std::vector<LineInfo>& lineInfo, std::vector<osg::Vec2>& vertices, bool useVerticalLayout)
{
   bool success = false;
   _hasCharCodes = true;

   unsigned int textLength = (_text != NULL ? strlen(_text) : 0);

   if ((textLength > 0) && (glyphInfoContainer != NULL) && _has_position && _textConfigBase.valid())
   {
      ComputeVerticesInput input;
      input._layout = _textConfigBase->getLayout();
      if ((input._layout == osgText::TextBase::VERTICAL) && !useVerticalLayout)
      {
         input._layout = osgText::TextBase::LEFT_TO_RIGHT;
      }

      //Split multi-line string into multiple strings
      std::vector<std::string> utf8Strings;
      input._lineInfoSize = splitIntoStringPerLine(_text, utf8Strings);
      input._charCodeOrGlyphCount = textLength;
      input._utf8Strings = &utf8Strings;
      input._glyphInfoVector = NULL;

      //Compute vertices and position offset
      success = computeGlyphVertices(input, glyphInfoContainer, positionOffset, lineInfo, vertices);

      success = true;
   }
   return success;
}

float TextSimpleBase::calcLeftOffset(LineInfo& lineInfo)
{
   float left_offset = _textConfigBase->getCharSize() * _charOffset.x();
   switch(_textConfigBase->getAlignment())
   {
   case osgText::TextBase::RIGHT_TOP:
   case osgText::TextBase::RIGHT_CENTER:
   case osgText::TextBase::RIGHT_BOTTOM:
   case osgText::TextBase::RIGHT_BASE_LINE:
   case osgText::TextBase::RIGHT_BOTTOM_BASE_LINE:
      left_offset -= lineInfo._xMax;
      break;
   case osgText::TextBase::CENTER_TOP:
   case osgText::TextBase::CENTER_CENTER:
   case osgText::TextBase::CENTER_BOTTOM:
   case osgText::TextBase::CENTER_BASE_LINE:
   case osgText::TextBase::CENTER_BOTTOM_BASE_LINE:
      left_offset -= (lineInfo._xMax + lineInfo._xMin) * 0.5f;
      break;
   case osgText::TextBase::LEFT_TOP:
   case osgText::TextBase::LEFT_CENTER:
   case osgText::TextBase::LEFT_BOTTOM:
   case osgText::TextBase::LEFT_BASE_LINE:
   case osgText::TextBase::LEFT_BOTTOM_BASE_LINE:
   default:
      left_offset -= lineInfo._xMin;
   }
   return left_offset;
}

float TextSimpleBase::calcBottomOffset(unsigned int lineCount)
{
   float characterHeight = _textConfigBase->getCharSize();

   float bottom_offset = characterHeight * _charOffset.y();
   switch(_textConfigBase->getAlignment())
   {
   case osgText::TextBase::LEFT_TOP:
   case osgText::TextBase::CENTER_TOP:
   case osgText::TextBase::RIGHT_TOP:
      bottom_offset -= _textBB.yMax();
      break;
   case osgText::TextBase::LEFT_CENTER:
   case osgText::TextBase::CENTER_CENTER:
   case osgText::TextBase::RIGHT_CENTER:
      bottom_offset -= ((_textBB.yMax()+_textBB.yMin()) * 0.5f);
      break;
   case osgText::TextBase::LEFT_BOTTOM:
   case osgText::TextBase::CENTER_BOTTOM:
   case osgText::TextBase::RIGHT_BOTTOM:
      bottom_offset -= _textBB.yMin();
      break;
   case osgText::TextBase::LEFT_BASE_LINE:
   case osgText::TextBase::CENTER_BASE_LINE:
   case osgText::TextBase::RIGHT_BASE_LINE:
      bottom_offset += _textConfigBase->getOutlineSize() * (characterHeight / static_cast<float>(_textConfigBase->getFontSize())) * 0.5f;
      break;
   case osgText::TextBase::LEFT_BOTTOM_BASE_LINE:
   case osgText::TextBase::CENTER_BOTTOM_BASE_LINE:
   case osgText::TextBase::RIGHT_BOTTOM_BASE_LINE:
   default:
      bottom_offset += (_textConfigBase->getOutlineSize() * (characterHeight / static_cast<float>(_textConfigBase->getFontSize())) * 0.5f)
                      - (characterHeight*(1.0f + _textConfigBase->getLineSpacing())*lineCount);
   }
   return bottom_offset;
}

void TextSimpleBase::addGlyphsToBatch()
{
   if (_textConfigBase.valid())
   {
      BatcherBase* batcherBase = _textConfigBase->getBatcher();
      if (batcherBase)
      {
         for (std::vector<unsigned int>::const_iterator itr = _charCodesOrGlyphIndices.begin(); itr != _charCodesOrGlyphIndices.end(); ++itr)
         {
            batcherBase->findOrAddGlyph(_hasCharCodes, *itr);
         }
      }
   }
}

void TextSimpleBase::addTextToBatch(float alpha, bool enableDepthTest, const osg::View* view, const FloatAndHalfFloat& vertexCompW) const
{
   BatcherBase* batcherBase = _textConfigBase->getBatcher();
   if (_glyphRepresentationCalculated && (batcherBase != NULL))
   {
      osg::ref_ptr<BatchElementContainer> batchElementContainer = new BatchElementContainer(BatchElementRenderInfo(_textConfigBase->getShaderType(), alpha, vertexCompW, enableDepthTest));
      batchElementContainer->_batchElements.push_back(new BatchElementTextBase(this, _textConfigBase->getColor(), _textConfigBase->getBackdropColor(), _numDrawableChars));
      batcherBase->addBatchElementContainer(batchElementContainer, _textConfigBase->getRenderBin(), view);
   }
}

void TextSimpleBase::addTextToBatch(const osg::ref_ptr<BatchElementContainer>& batchElementContainer, const osg::View* view) const
{
   BatcherBase* batcherBase = _textConfigBase->getBatcher();
   if (_glyphRepresentationCalculated && (batcherBase != NULL))
   {
      batcherBase->addBatchElementContainer(batchElementContainer, _textConfigBase->getRenderBin(), view);
   }
}

void TextSimpleBase::setCharOffset(const osg::Vec2& charOffset)
{
   if (_glyphRepresentationCalculated)
   {
      osg::Vec2 charOffsetDiffScaled = (charOffset - _charOffset) * getCharSize();
      for (unsigned int countVert = 0; countVert < _vertices.size(); ++countVert)
      {
         _vertices[countVert].x() += charOffsetDiffScaled.x();
         _vertices[countVert].y() += charOffsetDiffScaled.y();
      }
   }
   _charOffset = charOffset;
}

void TextSimpleBase::rotateChars(const std::vector<osg::Vec2>& vertices2D, const std::vector<osg::Vec3>& directions, const std::vector<osg::Vec2>& positionOffset, const osg::Vec3& normUpVec, float pitchAng)
{
   float cos_pitchAng = cosf(pitchAng);
   float sin_pitchAng = sinf(pitchAng);
   osg::Vec3 groundVec;
   osg::Vec3 heightVec;
   osg::Vec3 baseVec;

   _vertices.clear();
   _vertices.reserve(vertices2D.size());

   std::vector<osg::Vec2>::const_iterator itrV = vertices2D.begin();

   for (unsigned int currentChar = 0; currentChar < vertices2D.size()/4u; ++currentChar)
   {
      baseVec = directions[currentChar];
      baseVec.normalize();
      groundVec = normUpVec ^ baseVec;
      heightVec = (groundVec * cos_pitchAng) + (normUpVec * sin_pitchAng);
      for (unsigned int k = 0; k < 4; ++k)
      {
         osg::Vec3 vertex((baseVec * itrV->x()) + (heightVec * (itrV->y() + positionOffset[currentChar].y())));
         _vertices.push_back(osg::Vec3h(vertex.x(), vertex.y(), vertex.z()));
         ++itrV;
      }
   }
}

void TextSimpleBase::setBatcher(BatcherBase* batcherBase)
{
   if (_textConfigBase.valid())
   {
      _textConfigBase->setBatcher(batcherBase);
      addGlyphsToBatch();
   }
}

BatcherBase* TextSimpleBase::getBatcher() const
{
   return _textConfigBase.valid() ? _textConfigBase->getBatcher() : NULL;
}

